/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2019, 2021-2025 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <string.h>
#include <uchar.h>
#include "akfnetz.h"
#include "Entitaet.h"

static char32_t nummerisch (const char *);
static char32_t namentlich (const char *);
static int Vergleich (const void *, const void *);


extern char32_t
akfnetz_Entitaet (const char *Name)
{
  if (*Name == '&')
    ++Name;

  return ((*Name == '#') ? nummerisch (Name) : namentlich (Name));
}


static char32_t
nummerisch (const char *Name)
{
  char32_t e;
  const char *z = Name + 1;
  int Basis = 10;		// dezimal

  // hexadezimal?
  if ((*z | 0x20) == 'x')
    {
      Basis = 16;
      ++z;
    }

  e = (char32_t) strtoul (z, NULL, Basis);

  if (0x80 <= e && e <= 0x9F)	// Codepage 1252 :-(
    e = akfnetz_cp1252 (e);
  else if (e > 0x10FFFF || e < 0x20 || (e & 0xFFFE) == 0xFFFE
	   || (0xFDD0 <= e && e <= 0xFDEF) || (0xD800 <= e && e <= 0xDFFF))
    e = ERSATZZEICHEN;

  // ungueltig, wenn ausserhalb, Steuerzeichen, noncharacters,
  // UTF16-Ersatzcode

  return e;
}


static int
Vergleich (const void *n1, const void *n2)
{
  const struct Entitaetsnamen *a = n1, *b = n2;
  return strcmp (a->Name, b->Name);
}


static char32_t
namentlich (const char *Name)
{
  struct Entitaetsnamen Suche, *e;
  char n[40], *p;

  strncpy (n, Name, sizeof (n));
  n[sizeof (n) - 1] = '\0';
  p = strpbrk (n, ";& \t\r\n>\"'");
  if (p)
    *p = '\0';

  Suche.Name = n;
  Suche.Zeichen = 0;

  e = bsearch (&Suche, Entitaetsname, ENTITAETSANZAHL,
	       sizeof (Entitaetsname[0]), &Vergleich);

  return (e ? e->Zeichen : ERSATZZEICHEN);
}
