/* SPDX-License-Identifier: GPL-3.0-or-later */
/*
 * Copyright (c) 2021 Andreas K. Foerster <akf@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>		// SIZE_MAX
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "akfnetz.h"
#include "akfwebserver.h"

static int eingeblendet (FILE *, int, off_t, off_t);
static int gepuffert (FILE *, int, off_t, off_t);

#define PUFFERGROESSE 8192

#if defined(__linux__)
#include <sys/sendfile.h>

// Optimiert fuer GNU/Linux
extern int
Dateiinhalt (FILE * aus, int Datei, off_t Anfang, off_t Datenmenge)
{
  if (Datenmenge <= 0)
    return 0;

  fflush (aus);
  int Ausgabe = fileno (aus);
  off_t Position = Anfang, Rest = Datenmenge;

  while (Rest > 0)
    {
      ssize_t n;

      n = sendfile (Ausgabe, Datei, &Position,
		    (size_t) MIN (Rest, 0x7ffff000));

      if (n < 0)
	{
	  if (Rest == Datenmenge && (errno == ENOSYS || errno == EINVAL))
	    return eingeblendet (aus, Datei, Anfang, Datenmenge);

	  if (redsam (2))
	    akfnetz_Logbuch ("* %s\n\n", Fehlermeldung (errno));

	  return -1;
	}

      Rest -= n;
    }

  return 0;
}

#elif defined(__FreeBSD__)
#include <sys/socket.h>

// Optimiert fuer FreeBSD
extern int
Dateiinhalt (FILE * aus, int Datei, off_t Anfang, off_t Datenmenge)
{
  if (Datenmenge <= 0)
    return 0;

  fflush (aus);
  int Ausgabe = fileno (aus);
  off_t Position = Anfang, Rest = Datenmenge;

  while (Rest > 0)
    {
      off_t n;

      if (sendfile (Datei, Ausgabe, Position, Rest, NULL, &n, 0) < 0)
	{
	  if (Rest == Datenmenge && (errno == EOPNOTSUPP || errno == EINVAL))
	    return eingeblendet (aus, Datei, Anfang, Datenmenge);

	  if (redsam (2))
	    akfnetz_Logbuch ("* %s\n\n", Fehlermeldung (errno));

	  return -1;
	}

      Position += n;
      Rest -= n;
    }

  return 0;
}

#else // nicht systemspezifisch

extern int
Dateiinhalt (FILE * aus, int Datei, off_t Anfang, off_t Datenmenge)
{
  return eingeblendet (aus, Datei, Anfang, Datenmenge);
}

#endif


#if defined(_POSIX_MAPPED_FILES) && _POSIX_MAPPED_FILES > 0
#include <sys/mman.h>

// Inhalt per mmap einblenden
static int
eingeblendet (FILE * aus, int Datei, off_t Anfang, off_t Datenmenge)
{
  if (Datenmenge <= 0)
    return 0;

  off_t Versatz = 0;

  if (Anfang > 0xFFFF)
    {
      size_t page_size = (size_t) sysconf (_SC_PAGESIZE);
      Versatz = Anfang & ~(page_size - 1);
    }

  off_t Groesse = (Anfang - Versatz) + Datenmenge;
  if (Groesse > SIZE_MAX)
    return gepuffert (aus, Datei, Anfang, Datenmenge);

  char *Daten = mmap (NULL, Groesse, PROT_READ, MAP_PRIVATE, Datei, Versatz);

  if (Daten == MAP_FAILED)
    return gepuffert (aus, Datei, Anfang, Datenmenge);

  posix_madvise (Daten, Groesse, POSIX_MADV_SEQUENTIAL);

  int e = 0;
  if (fwrite (Daten + (Anfang - Versatz), 1, Datenmenge, aus) < Datenmenge)
    {
      e = -1;
      if (redsam (2))
	akfnetz_Logbuch ("* %s\n\n", Fehlermeldung (errno));
    }

  munmap (Daten, Groesse);

  return e;
}

#else // mmap nicht unterstuetzt

static int
eingeblendet (FILE * aus, int Datei, off_t Anfang, off_t Datenmenge)
{
  return gepuffert (aus, Datei, Anfang, Datenmenge);
}

#endif


// Inhalt gepuffert uebertragen
static int
gepuffert (FILE * aus, int Datei, off_t Anfang, off_t Datenmenge)
{
  if (Datenmenge <= 0)
    return 0;

  char *Puffer = malloc (PUFFERGROESSE);
  if (!Puffer)
    return -1;

  if (Anfang > 0)
    lseek (Datei, Anfang, SEEK_SET);

  int e = 0;
  off_t Rest = Datenmenge;

  while (Rest > 0)
    {
      ssize_t n;

      do
	n = read (Datei, Puffer, PUFFERGROESSE);
      while (n < 0 && errno == EINTR);

      if (n < 0)
	{
	  e = -1;
	  if (redsam (2))
	    akfnetz_Logbuch ("* %s\n\n", Fehlermeldung (errno));
	  break;
	}

      if (!n || fwrite (Puffer, 1, n, aus) < n)
	{
	  e = -1;
	  break;
	}

      Rest -= n;
    }

  free (Puffer);

  return e;
}
