#include <zmodem.h>
#include <zmem.h>
#include <filexfer.h>

/*
	Tom Jennings
	Fido Software
	12 July 90

	Zmodem send/receive base routines

	Derived from the Zmodem protocol files
	Chuck Forsberg, Omen Technology Inc
	05-09-88

This will accept up to 8192 byte data packets, generated by
BinkleyTerm and Opus.


File transfer protocol support routines needed, besides
the usual:

xferstat(n,z,0,s)Controls error reporting and status logging for
int n;		file transfers. N is the function call number;
long z;
char *s;	0 == Initialize: clears the file status counters
		totl_???? to zero, and readies the display.

		1 == start new file: Displays the passed filename (s)
		and overall file size (z).

		2 == status display: current block number, etc. Z
		is current byte offset. s is a runtime message.

		3 == error message: similar to above but can be
		handled differently if necessary.

badname(s)	Returns true if the passed ASCII name is a device
		name.

baudrate()	Returns the current baud rate.

fexist(fn)	Returns true if this file cannot be overwritten;
char *fn;	ie. if it exists and overwriting is not allowed by
		the program.

chkabort()	Checks the local keyboard and returns true if the
		file transfer should be aborted.
*/

char abortstr[]= {24,24,24,24,24,24,24,24,24,24,8,8,8,8,8,8,8,8,8,8,0};

char *Zendnames[] = {
	"ZCRCE",
	"ZCRCG",
	"ZCRCQ",
	"ZCRCW"
};
char *framenames[] = {
	"ZRQINIT",
	"ZRINIT",
	"ZSINIT",
	"ZACK",
	"ZFILE",
	"ZSKIP",
	"ZNAK",
	"ZABORT",
	"ZFIN",
	"ZRPOS",
	"ZDATA",
	"ZEOF",
	"ZFERR",
	"ZCRC",
	"ZCHLNGE",
	"ZCOMPL",
	"ZCAN",
	"ZFREECNT",
	"ZCOMMAND",
	"ZSTDERR",

	"TIMEOUT"	/* a fake value for error reports */
};

char *namelist;			/* list of received filenames */
unsigned listlen;		/* its max. length */

int rxtimeout;			/* Hundredths of seconds to wait for something */
int rxframeind;			/* ZBIN ZBIN32, or ZHEX type of frame received */
int rxtype;			/* Type of header received */
int rxcount;			/* Count of data bytes received */
char rxhdr[4];			/* Received header */
char txhdr[4];			/* Transmitted header */
long rxbytes;			/* total received/acked bytes */
long rxpos;			/* Received file position */
long txpos;			/* Transmitted file position */
int txfcs32;			/* TRUE means send binary frames with 32 bit FCS */
int crc32t;			/* Display flag indicating 32 bit CRC being sent */
int znulls;			/* Number of nulls to send at beginning of ZDATA hdr */
char attn[ZATTNLEN+1];		/* Attention string Rx sends to Tx on err */
FLAG zmabort;			/* 1 == manual keyboard abort */

int tryzhdrtype;		/* Header type to send corresponding to Last rx close */

FLAG txwindow;			/* 1 == I do windows */
unsigned txwsize;		/* size of the transmitted window */
unsigned txwspace;		/* how often to send ZCRCQ */

long lrxpos;			/* Receiver's last reported offset */
int errors;

unsigned blklen;		/* length of transmitted records */
unsigned rxbuflen;		/* Receiver's max buffer length or 0 for accepts streaming */
FLAG eofseen;			/* EOF seen on input */
int rxflags;			/* receivers Rx flags (CANxxx bit masks) */
FLAG zctlesc;			/* escape control characters */
FLAG cantovio;			/* Rx can't overlap disk and serial IO */
long bytcnt;
char wantfcs32;			/* Tx want to send 32 bit FCS */

FLAG lzconv;			/* Tx ZMODEM file conversion request */
FLAG lzmanag;			/* Tx ZMODEM file management request */
FLAG lztrans;			/* Tx ZMODEM file transport request */

FLAG zconv;			/* Rx ZMODEM file conversion request */
FLAG zmanag;			/* Rx ZMODEM file management request */
FLAG ztrans;			/* Rx ZMODEM file transport request */

long lastsync;			/* Last offset to which we got a ZRPOS */
int beenhereb4;			/* How many times we've been ZRPOS'd same place */

FLAG lastsent;			/* Last char we sent */
FLAG Not8bit;			/* Seven bits seen on header */


/*
 *  Crc - 32 BIT ANSI X3.66 CRC checksum files
 */

/*
 * Copyright (C) 1986 Gary S. Brown.  You may use this program, or
 * code or tables extracted from it, as desired without restriction.
 */

/* First, the polynomial itself and its table of feedback terms.  The  */
/* polynomial is                                                       */
/* X^32+X^26+X^23+X^22+X^16+X^12+X^11+X^10+X^8+X^7+X^5+X^4+X^2+X^1+X^0 */
/* Note that we take it "backwards" and put the highest-order term in  */
/* the lowest-order bit.  The X^32 term is "implied"; the LSB is the   */
/* X^31 term, etc.  The X^0 term (usually shown as "+1") results in    */
/* the MSB being 1.                                                    */

/* Note that the usual hardware shift register implementation, which   */
/* is what we're using (we're merely optimizing it by doing eight-bit  */
/* chunks at a time) shifts bits into the lowest-order term.  In our   */
/* implementation, that means shifting towards the right.  Why do we   */
/* do it this way?  Because the calculated CRC must be transmitted in  */
/* order from highest-order term to lowest-order term.  UARTs transmit */
/* characters in order from LSB to MSB.  By storing the CRC this way,  */
/* we hand it to the UART in the order low-byte to high-byte; the UART */
/* sends each low-bit to hight-bit; and the result is transmission bit */
/* by bit from highest- to lowest-order term without requiring any bit */
/* shuffling on our part.  Reception works similarly.                  */

/* The feedback terms table consists of 256, 32-bit entries.  Notes:   */
/*                                                                     */
/*  1. The table can be generated at runtime if desired; code to do so */
/*     is shown later.  It might not be obvious, but the feedback      */
/*     terms simply represent the results of eight shift/xor opera-    */
/*     tions for all combinations of data and CRC register values.     */
/*                                                                     */
/*  2. The CRC accumulation logic is the same for all CRC polynomials, */
/*     be they sixteen or thirty-two bits wide.  You simply choose the */
/*     appropriate table.  Alternatively, because the table can be     */
/*     generated at runtime, you can start by generating the table for */
/*     the polynomial in question and use exactly the same "updcrc",   */
/*     if your application needn't simultaneously handle two CRC       */
/*     polynomials.  (Note, however, that XMODEM is strange.)          */
/*                                                                     */
/*  3. For 16-bit CRCs, the table entries need be only 16 bits wide;   */
/*     of course, 32-bit entries work OK if the high 16 bits are zero. */
/*                                                                     */
/*  4. The values must be right-shifted by eight bits by the "updcrc"  */
/*     logic; the shift must be unsigned (bring in zeroes).  On some   */
/*     hardware you could probably optimize the shift in assembler by  */
/*     using byte-swap instructions.                                   */

long crc32tab[] = { /* CRC polynomial 0xedb88320 */

0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988, 0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172, 0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924, 0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e, 0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0, 0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a, 0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc, 0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236, 0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38, 0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2, 0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94, 0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};

/* crctab calculated by Mark G. Mendel, Network Systems Corporation */

unsigned crctab[256] = {
    0x0000,  0x1021,  0x2042,  0x3063,  0x4084,  0x50a5,  0x60c6,  0x70e7,
    0x8108,  0x9129,  0xa14a,  0xb16b,  0xc18c,  0xd1ad,  0xe1ce,  0xf1ef,
    0x1231,  0x0210,  0x3273,  0x2252,  0x52b5,  0x4294,  0x72f7,  0x62d6,
    0x9339,  0x8318,  0xb37b,  0xa35a,  0xd3bd,  0xc39c,  0xf3ff,  0xe3de,
    0x2462,  0x3443,  0x0420,  0x1401,  0x64e6,  0x74c7,  0x44a4,  0x5485,
    0xa56a,  0xb54b,  0x8528,  0x9509,  0xe5ee,  0xf5cf,  0xc5ac,  0xd58d,
    0x3653,  0x2672,  0x1611,  0x0630,  0x76d7,  0x66f6,  0x5695,  0x46b4,
    0xb75b,  0xa77a,  0x9719,  0x8738,  0xf7df,  0xe7fe,  0xd79d,  0xc7bc,
    0x48c4,  0x58e5,  0x6886,  0x78a7,  0x0840,  0x1861,  0x2802,  0x3823,
    0xc9cc,  0xd9ed,  0xe98e,  0xf9af,  0x8948,  0x9969,  0xa90a,  0xb92b,
    0x5af5,  0x4ad4,  0x7ab7,  0x6a96,  0x1a71,  0x0a50,  0x3a33,  0x2a12,
    0xdbfd,  0xcbdc,  0xfbbf,  0xeb9e,  0x9b79,  0x8b58,  0xbb3b,  0xab1a,
    0x6ca6,  0x7c87,  0x4ce4,  0x5cc5,  0x2c22,  0x3c03,  0x0c60,  0x1c41,
    0xedae,  0xfd8f,  0xcdec,  0xddcd,  0xad2a,  0xbd0b,  0x8d68,  0x9d49,
    0x7e97,  0x6eb6,  0x5ed5,  0x4ef4,  0x3e13,  0x2e32,  0x1e51,  0x0e70,
    0xff9f,  0xefbe,  0xdfdd,  0xcffc,  0xbf1b,  0xaf3a,  0x9f59,  0x8f78,
    0x9188,  0x81a9,  0xb1ca,  0xa1eb,  0xd10c,  0xc12d,  0xf14e,  0xe16f,
    0x1080,  0x00a1,  0x30c2,  0x20e3,  0x5004,  0x4025,  0x7046,  0x6067,
    0x83b9,  0x9398,  0xa3fb,  0xb3da,  0xc33d,  0xd31c,  0xe37f,  0xf35e,
    0x02b1,  0x1290,  0x22f3,  0x32d2,  0x4235,  0x5214,  0x6277,  0x7256,
    0xb5ea,  0xa5cb,  0x95a8,  0x8589,  0xf56e,  0xe54f,  0xd52c,  0xc50d,
    0x34e2,  0x24c3,  0x14a0,  0x0481,  0x7466,  0x6447,  0x5424,  0x4405,
    0xa7db,  0xb7fa,  0x8799,  0x97b8,  0xe75f,  0xf77e,  0xc71d,  0xd73c,
    0x26d3,  0x36f2,  0x0691,  0x16b0,  0x6657,  0x7676,  0x4615,  0x5634,
    0xd94c,  0xc96d,  0xf90e,  0xe92f,  0x99c8,  0x89e9,  0xb98a,  0xa9ab,
    0x5844,  0x4865,  0x7806,  0x6827,  0x18c0,  0x08e1,  0x3882,  0x28a3,
    0xcb7d,  0xdb5c,  0xeb3f,  0xfb1e,  0x8bf9,  0x9bd8,  0xabbb,  0xbb9a,
    0x4a75,  0x5a54,  0x6a37,  0x7a16,  0x0af1,  0x1ad0,  0x2ab3,  0x3a92,
    0xfd2e,  0xed0f,  0xdd6c,  0xcd4d,  0xbdaa,  0xad8b,  0x9de8,  0x8dc9,
    0x7c26,  0x6c07,  0x5c64,  0x4c45,  0x3ca2,  0x2c83,  0x1ce0,  0x0cc1,
    0xef1f,  0xff3e,  0xcf5d,  0xdf7c,  0xaf9b,  0xbfba,  0x8fd9,  0x9ff8,
    0x6e17,  0x7e36,  0x4e55,  0x5e74,  0x2e93,  0x3eb2,  0x0ed1,  0x1ef0
};

/*
 * updcrc macro derived from article Copyright (C) 1986 Stephen Satchell. 
 *  NOTE: First argument must be in range 0 to 255.
 *        Second argument is referenced twice.
 * 
 * Programmers may incorporate any or all code into their programs, 
 * giving proper credit within the source. Publication of the 
 * source routines is permitted so long as proper credit is given 
 * to Stephen Satchell, Satchell Evaluations and Chuck Forsberg, 
 * Omen Technology.
 */

/* This function is never called; it is to force the linker to load 
ZRECV and ZSEND when module ZMODEM (this one) is referred to. Fido et al
can still name and position the modules explicitly. */

static link_em_up() {

	zrecv();
	zsend();
}

/* Manual abort: return true if either the abort flag is set or
someone hits ESCape. */

zabort() {

	return(zmabort ? 1 : (zmabort= chkabort()));
}

/* Send ZMODEM binary header hdr of type type */

zsbhdr(type, hdr)
char *hdr;
{
int n;
unsigned crc;

	if (type == ZDATA) {
		for (n= znulls; --n >= 0; )
			modout(0);
	}
	modout(ZPAD); modout(ZDLE);

	if (crc32t= txfcs32) zsbh32(hdr, type);
	else {
		modout(ZBIN); zmodout(type); crc= updcrc(type, 0);

		for (n=4; --n >= 0; ++hdr) {
			zmodout(*hdr);
			crc= updcrc((0377 & *hdr), crc);
		}
		crc= updcrc(0,updcrc(0,crc));
		zmodout(crc >> 8);
		zmodout(crc);
	}
}


/* Send ZMODEM binary header hdr of type type */

zsbh32(hdr, type)
char *hdr;
{
int n;
long crc;

	modout(ZBIN32);  zmodout(type);
	crc= 0xFFFFFFFFL; crc= UPDC32(type, crc);

	for (n=4; --n >= 0; ++hdr) {
		crc= UPDC32((0377 & *hdr), crc);
		zmodout(*hdr);
	}
	crc= ~crc;
	for (n=4; --n >= 0;) {
		zmodout((int)crc);
		crc >>= 8;
	}
}

/* Send ZMODEM HEX header hdr of type type */

zshhdr(type, hdr)
char *hdr;
{
int n;
unsigned crc;

	modout(ZPAD); modout(ZPAD); modout(ZDLE); modout(ZHEX);
	zputhex(type);
	crc32t= 0;

	crc= updcrc(type, 0);
	for (n=4; --n >= 0; ++hdr) {
		zputhex(*hdr); crc= updcrc((0377 & *hdr), crc);
	}
	crc= updcrc(0,updcrc(0,crc));
	zputhex(crc>>8); zputhex(crc);

	/* Make it printable on remote machine */

	modout(015); modout(0212);
	/*
	 * Uncork the remote in case a fake XOFF has stopped data flow
	 */
	if (type != ZFIN && type != ZACK)
		modout(021);
}

/*
 * Send binary array buf of length length, with ending ZDLE sequence frameend
 */

zsdata(buf, length, frameend)
char *buf;
{
unsigned crc;

	if (crc32t) zsda32(buf, length, frameend);

	else {
		crc= 0;
		for (;--length >= 0; ++buf) {
			zmodout(*buf); crc= updcrc((0377 & *buf), crc);
		}
		modout(ZDLE); modout(frameend);
		crc= updcrc(frameend, crc);

		crc= updcrc(0,updcrc(0,crc));
		zmodout(crc>>8); zmodout(crc);
	}
	if (frameend == ZCRCW) modout(XON);  
}

zsda32(buf, length, frameend)
char *buf;
{
int c;
long crc;

	crc= 0xFFFFFFFFL;
	for (;--length >= 0; ++buf) {
		c= *buf & 0377;
		if (c & 0140)
			modout(lastsent= c);
		else
			zmodout(c);
		crc= UPDC32(c, crc);
	}
	modout(ZDLE); modout(frameend);
	crc= UPDC32(frameend, crc);

	crc= ~crc;
	for (length=4; --length >= 0;) {
		zmodout((int)crc);  crc >>= 8;
	}
}

/*
 * Receive array buf of max length with ending ZDLE sequence
 *  and CRC.  Returns the ending character or error code.
 *  NB: On errors may store length+1 bytes!
 */

zrdata(buf, length)
char *buf;
unsigned length;
{
int c;
unsigned crc;
char *end;
int d;

	if (rxframeind == ZBIN32)
		return zrdat32(buf, length);

	crc= rxcount= 0;  end= buf + length;
	while (buf <= end) {
		if ((c= zdlread()) & ~0377) {
crcfoo:			switch (c) {
			case GOTCRCE:
			case GOTCRCG:
			case GOTCRCQ:
			case GOTCRCW:
				d= c;  c &= 0377;
				crc= updcrc(c, crc);
				if ((c= zdlread()) & ~0377) goto crcfoo;

				crc= updcrc(c, crc); 
				if ((c= zdlread()) & ~0377) goto crcfoo;

				crc= updcrc(c, crc);
				if (crc & 0xFFFF) {
					xferstat(3,rxpos,0,"Bad Data CRC");
					return ERROR;
				}
				rxcount= length - (end - buf);
				return d;

			case GOTCAN:
				xferstat(3,rxpos,0,"Sender Cancelled");
				return ZCAN;

			case TIMEOUT:
				xferstat(3,rxpos,0,"Timeout");
				return c;

			default:
				xferstat(3,rxpos,0,"Bad data packet");
				return c;
			}
		}
		*buf++= c;
		crc= updcrc(c, crc);
	}
	xferstat(3,rxpos,0,"Data packet too long");
	return ERROR;
}

zrdat32(buf,length)
char *buf;
unsigned length;
{
int c;
long crc;
char *end;
int d;

	crc= 0xFFFFFFFFL;  rxcount= 0;  end= buf + length;
	while (buf <= end) {
		if ((c= zdlread()) & ~0377) {
crcfoo:			switch (c) {
			case GOTCRCE:
			case GOTCRCG:
			case GOTCRCQ:
			case GOTCRCW:
				d= c;  c &= 0377;
				crc= UPDC32(c, crc);
				if ((c= zdlread()) & ~0377) goto crcfoo;

				crc= UPDC32(c, crc);
				if ((c= zdlread()) & ~0377) goto crcfoo;

				crc= UPDC32(c, crc);
				if ((c= zdlread()) & ~0377) goto crcfoo;

				crc= UPDC32(c, crc);
				if ((c= zdlread()) & ~0377) goto crcfoo;

				crc= UPDC32(c, crc);
				if (crc != 0xDEBB20E3) {
					xferstat(3,rxpos,0,"Bad Data CRC32");
					return ERROR;
				}
				rxcount= length - (end - buf);
				return d;

			case GOTCAN:
				xferstat(3,rxpos,0,"Sender Cancelled32");
				return ZCAN;

			case TIMEOUT:
				xferstat(3,rxpos,0,"Timeout32");
				return c;
			default:
				xferstat(3,rxpos,0,"Bad data packet32");
				return c;
			}
		}
		*buf++= c;
		crc= UPDC32(c, crc);
	}
	xferstat(3,rxpos,0,"Data packet too long32");
	return ERROR;
}

/* bugc(f)
int f;
{
	place(line,column);
	if (f == -1) { ansiout('_'); return;}
	if (f > 126) { ansiout('!'); f &= 0x7f;}
	if (f < ' ') { ansiout('^'); ansiout(f + '@');}
	else ansiout(f);
}

bugs(s)
char *s;
{
	place(line,column);
	while (*s) ansiout(*s++);
}
*/

/*
 * Read a ZMODEM header to hdr, either binary or hex.
 *  eflag controls local display of non zmodem characters:
 *	0:  no display
 *	1:  display printing characters only
 *	2:  display all non ZMODEM characters
 *  On success, set Zmodem to 1, set rxpos and return type of header.
 *   Otherwise return negative on error.
 *   Return ERROR instantly if ZCRCW sequence, for fast error recovery.
 */

zgethdr(hdr, eflag)
char *hdr;
{
int c;
unsigned crapcount, cancount;

/**/	crapcount= -1;	/* lots */

/*	crapcount= zmblkmax + baudrate();	*/	/* max chars til next blk start */

	rxframeind= rxtype= 0;

startover:
	cancount= 5;					/* CANcel char counter */
again:	c= modin(rxtimeout);				/* get a character */
	switch (c) {
		case ZPAD | 0200: Not8bit= c;		/* huh? */
		case ZPAD: break;			/* This is what we want. */
		case TIMEOUT: return TIMEOUT;		/* we got nothing */

/* General CAN and garbage character handler */

		case CAN:				/* watch for cancel */
gotcan:			if (--cancount <= 0) return ZCAN; /* canceled */
			switch (c= modin(100)) {
				case TIMEOUT: goto again; /* spurious CAN? */
				case ZCRCW: return ERROR;
				case CAN: if (--cancount <= 0) return ZCAN;
					goto again;
				default: break;
			}			/* **** FALL THRU TO **** */
		default:
garbage:
/**/ /* Problems with this, whats the right number? */

/*		if ( --crapcount == 0) {
				xferstat(3,rxpos,0,"Excessive line noise!");
				return(ERROR);
			}
*/			goto startover;
	}

/* Got our ZPAD character. Now we want a ZDLE, which preceeds the type. */

	cancount= 5;
splat:	c= noxrd7();
	switch (c) {
		case ZDLE: break;		/* This is what we want. */
		case ZPAD: goto splat;		/* skip extra pad chars */
		case TIMEOUT: return TIMEOUT;	/* got nothing */
		default: goto garbage;		/* garbage characters */
	}

/* We had a ZPAD, then a ZDLE, now we should have a header TYPE. */

	c= noxrd7();
	switch (c) {
		case ZBIN:
			rxframeind= ZBIN;
			c= zrbhdr(hdr);
			break;

		case ZBIN32:
			rxframeind= ZBIN32;
			c= zrbhdr32(hdr);
			break;

		case ZHEX:
			rxframeind= ZHEX; 
			c= zrhhdr(hdr);
			break;

		case TIMEOUT: return TIMEOUT;
		case CAN: goto gotcan;

		default: goto garbage;
	}
	rxpos= hdr[ZP3] & 0377;
	rxpos= (rxpos << 8) + (hdr[ZP2] & 0377);
	rxpos= (rxpos << 8) + (hdr[ZP1] & 0377);
	rxpos= (rxpos << 8) + (hdr[ZP0] & 0377);

	return c;
}

/* Receive a binary style header (type and position) */

zrbhdr(hdr)
char *hdr;
{
int c, n;
unsigned crc;

	if ((c= zdlread()) & ~0377) return c;		/* timeout or special */
	rxtype= c;
	crc= updcrc(c,0);

	for (n=4; --n >= 0; ++hdr) {
		if ((c= zdlread()) & ~0377)
			return c;
		crc= updcrc(c, crc);
		*hdr= c;
	}

	if ((c= zdlread()) & ~0377) return c;
	crc= updcrc(c, crc);				/* do the two CRC bytes */
	if ((c= zdlread()) & ~0377) return c;
	crc= updcrc(c, crc);

	if (crc & 0xFFFF) {
		xferstat(3,rxpos,0,"Bad Bin Hdr CRC");
		return ERROR;
	}
	return rxtype;
}

/* Receive a binary style header (type and position) with 32 bit FCS */

zrbhdr32(hdr)
char *hdr;
{
int c, n;
long crc;

	if ((c= zdlread()) & ~0377)
		return c;
	rxtype= c;
	crc= 0xFFFFFFFFL; crc= UPDC32(c, crc);

	for (n= 4; --n >= 0; ++hdr) {
		if ((c= zdlread()) & ~0377)
			return c;
		crc= UPDC32(c, crc);
		*hdr= c;
	}
	for (n= 4; --n >= 0;) {
		if ((c= zdlread()) & ~0377)
			return c;
		crc= UPDC32(c, crc);
	}
	if (crc != 0xDEBB20E3) {
		xferstat(3,rxpos,0,"Bad Bin Hdr CRC32");
		return ERROR;
	}
	return rxtype;
}

/* Receive a hex style header (type and position) */

zrhhdr(hdr)
char *hdr;
{
int c;
unsigned crc;
int n;

	if ((c= zgethex()) < 0) return c;
	rxtype= c;
	crc= updcrc(c, 0);

	for (n=4; --n >= 0; ++hdr) {
		if ((c= zgethex()) < 0) return c;
		crc= updcrc(c, crc);
		*hdr= c;
	}
	if ((c= zgethex()) < 0) return c;
	crc= updcrc(c, crc);			/* get CRC bytes */
	if ((c= zgethex()) < 0) return c;
	crc= updcrc(c, crc);

	if (crc & 0xFFFF) {
		xferstat(3,rxpos,0,"Bad Hex CRC");
		return ERROR;
	}
	switch ( c= modin(10)) {		/* flush CR/LFs etc */
		case 0215: Not8bit= c; 		/* fall through ... */
		case 015: if ((c= modin(10)) == 012) Not8bit |= c;
	}
	return rxtype;
}

/* Send a byte as two hex digits */

zputhex(c)
int c;
{
static char digits[]= "0123456789abcdef";

	modout(digits[(c&0xF0)>>4]);
	modout(digits[(c)&0xF]);
}

/*
 * Send character c with ZMODEM escape sequence encoding.
 *  Escape XON, XOFF. Escape CR following @ (Telenet net escape)
 */

zmodout(c)
{

	if (c & 0140) modout(lastsent= c);
	else {
		switch (c &= 0377) {
		case ZDLE:
			modout(ZDLE);
			modout (lastsent= (c ^= 0100));
			break;

		case 015:
		case 0215:
			if (!zctlesc && (lastsent & 0177) != '@') goto sendit;
			/* **** FALL THRU TO **** */

		case 020:
		case 021:
		case 023:
		case 0220:
		case 0221:
		case 0223:
			modout(ZDLE);
			c ^= 0100;
	sendit:		modout(lastsent= c);
			break;
		default:
			if (zctlesc && !(c & 0140)) {
				modout(ZDLE);
				c ^= 0100;
			}
			modout(lastsent= c);
		}
	}
}

/* Decode two lower case hex digits into an 8 bit byte value */

zgethex()
{
int c, n;

	if ((c= noxrd7()) < 0) return c;		/* TIMEOUT, ERROR */

	n= c - '0';
	if (n > 9) n -= ('a' - ':');
	if (n & ~0x0F) return ERROR;

	if ((c= noxrd7()) < 0) return c;		/* 2nd char is bad */

	c -= '0';
	if (c > 9) c -= ('a' - ':');
	if (c & ~0xF) return ERROR;
	c += (n<<4);
	return c;
}

/*
 * Read a byte, checking for ZMODEM escape encoding
 *  including CAN*5 which represents a quick abort
 */
zdlread() {
int c;

again:	c= modin(rxtimeout);
	if (c < 0) return(TIMEOUT);
	if (c & 0140) return(c);		/* accept printable */

	switch (c) {
		case ZDLE: break;		/* acceptable */

		case 023:			/* always reject */
		case 0223:
		case 021:
		case 0221: goto again;

		default:			/* filter control chars */
			if (zctlesc && !(c & 0140)) goto again;
			return c;
	}

/* We got a ZDLE (aka CAN); see what comes next, if any */

again2:
	if ((c= modin(rxtimeout)) < 0) 	return c;		/* timeout */
	if (c == CAN) if ((c= modin(rxtimeout)) < 0) return c;	/* CAN #2 */
	if (c == CAN) if ((c= modin(rxtimeout)) < 0) return c;	/* CAN #3 */
	if (c == CAN) if ((c= modin(rxtimeout)) < 0) return c;	/* CAN #4 */

	switch (c) {
		case CAN: return GOTCAN;		/* CAN #5, cancel */

		case ZCRCE:
		case ZCRCG:
		case ZCRCQ:
		case ZCRCW: return (c | GOTOR);

		case ZRUB0: return 0177;		/* escaped rubouts */
		case ZRUB1: return 0377;

		case 023:
		case 0223:
		case 021:
		case 0221: goto again2;			/* always ignore these */

		default: if (zctlesc && !(c & 0140)) 
				goto again2;		/* filter control chars */
			if ((c & 0140) == 0100) 
				return (c ^ 0100);	/* controllify @ char */
			break;
	}
	return ERROR;
}

/*
 * Read a character from the modem line with timeout.
 *  Eat parity, XON and XOFF characters.
 */

noxrd7() {
int c;

	while (1) {
		if ((c= modin(rxtimeout)) < 0) return(c);
		switch (c &= 0177) {
			case XON:		/* always ignore these, */
			case XOFF:
				continue;

			case '\r':		/* always acceptable */
			case '\n':		/* control chars, */
			case ZDLE:
				return c;

			default:		/* filter out control chars */
				if (zctlesc && !(c & 0140)) continue;
				return c;
		}
	}
}

/* Store long integer pos in txhdr */

stohdr(pos)
long pos;
{
	txhdr[ZP0]= pos;
	txhdr[ZP1]= pos>>8;
	txhdr[ZP2]= pos>>16;
	txhdr[ZP3]= pos>>24;
}

/* Recover a long integer from a header */

long
rclhdr(hdr)
char *hdr;
{
long l;

	l= (hdr[ZP3] & 0377);
	l= (l << 8) | (hdr[ZP2] & 0377);
	l= (l << 8) | (hdr[ZP1] & 0377);
	l= (l << 8) | (hdr[ZP0] & 0377);
	return l;
}

/* Convert MSDOS packed time to long seconds since 1970. (Basic conversion
derived from code supplied by Stuart Gathman.) */

long
dos2sec(dostime,dosdate)
unsigned dostime,dosdate;
{
int y,dayofyear;
long time;

#define Uyear ((dosdate >> 9) & 0x3f)		/* as stored in search */
#define Umonth ((dosdate >> 5) & 0x0f)		/* first/next structure */
#define Uday (dosdate & 0x1f)			/* or get file time DOS call */
#define Uhour ((dostime >> 11) & 0x1f)
#define Uminute ((dostime >> 5) & 0x3f)
#define Usecond ((dostime & 0x1f) << 1)		/* (kept as 2 second resolution) */

	y= Uyear + 80; 				/* DOS date base is 1980 */
	dayofyear= Uday + ((Umonth * 275) / 9) - 30;/* day of year, */
	if (Umonth > 2) {			/* handle leapyears */
		--dayofyear;			/* Feb 28/29, 1 Mar */
		if (Uyear % 4) --dayofyear;
	}
	time= dayofyear + ((long)(y - 1) * 1461L) / 4; /* day of century */

	time -= 25203L;				/* minus days 1900 - 1970 */

	time *= 86400L;				/* second of century */
	time += (long)Uhour * 3600L;		/* add seconds this day, */
	time += Uminute * 60;
	time += Usecond;
/*	cprintf("dos2sec: %02d/%02d/%02d %02d:%02d:%02d=%lu\r\n",y,Umonth,Uday,Uhour,Uminute,Usecond,time);
*/	return(time);
}

/* Convert seconds since 1970 to MSDOS packed time and date. The DOS date is
in the upper two bytes, time in the lower. (Same as passed in a TELINK 
block.) (Basic conversion derived from code supplied by Stuart Gathman.) */

long
sec2dos(sec70)
long sec70;
{
long time;
int dayofyear,leapyear;
int year,month,day;

	time= sec70 % 86400L;			/* time only */
	sec70 /= 86400L;			/* date only */
	sec70 += 25203L;			/* add days 1900 - 1970 */
	leapyear= 2;

	year= ((sec70 - (sec70 / 1461) + 364) / 365);	/* make year, */
	dayofyear= sec70 - ((long)(year - 1) * 1461) / 4;/* day in current year */
	if (year % 4 == 0) leapyear= 1;
	if (dayofyear > 59 && ((dayofyear > 60) || (leapyear == 2)))
		dayofyear += leapyear;
	month= (269 + (dayofyear * 9)) / 275;
	day= dayofyear + 30 - ((275 * month) / 9);

/*	cprintf("\r\nsec2dos: %lu == %02d/%02d/%02d %02d:%02d:%02d\r\n",x,year,month,day,time / 3600L,(time % 3600L) / 60L,(time % 3600L) % 60L);
*/
	sec70= 0L;
	sec70 |= ((year - 80) & 0x3f) << 9;	/* pack the date (starts 1980) */
	sec70 |= (month & 0x0f) << 5;		/* into lower 16 bits */
	sec70 |= day & 0x1f;
	sec70 <<= 16L;				/* shift to upper 16 bits */

	sec70 |= ((time / 3600L) & 0x1f) << 11;	/* then pack the time */
	time %= 3600L;				/* hours, */
	sec70 |= ((time / 60L) & 0x3f) << 5;	/* minutes, */
	sec70 |= ((time % 60L) & 0x1f) >> 1;	/* seconds. */
	return(sec70);
}

/* send cancel string to get the other end to shut up */

aborttx() {
char *cp;

	flush();			/* pitch everything */
	for (cp= abortstr; *cp; ) 	/* output it */
		modout(*cp++);	
}
