// Copyright (c) 1997, Matthew E. Kimmel
//
// This class emulates a text screen (modeled on a DOS machine in
// text mode) which can print text and scroll up or down.
package kimmel.awt;

import java.awt.*;
import java.util.*;
import java.awt.image.*;

/** The TextScreen class is a widget which provides the functionality
    of a text screen (a la DOS in text mode), which can print text
    and optionally scroll up or down.  Methods are also included to
    read and optionally echo keyboard input.  It is derived from
    java.awt.Canvas.  This component is designed for use with
    fixed-width fonts and may look a bit strange with proportional
    fonts.

    A TextScreen widget may have various attributes associated
    with it.  These attributes may be set with the constructor
    or with the setAttributes method, and may be retrieved with
    the getAttributes method.  Attributes are int constants which
    are OR'd together to form an attribute word.  Attributes
    include:
        TextScreen.KEYECHO - Echo keyboard input during reads.
        TextScreen.CURSOR - Display a cursor
        TextScreen.AUTOWRAP - Wrap at end of line--code is somewhat crude.

    Note that, for performance reasons, this widget does not
    automatically wrap at end of line.
    
    The widget also supports the concept of a Region.  If a Region
    is set, certain operations only affect that region of the screen.

    @see java.awt.Canvas
    @author Matthew E. Kimmel
*/
public class TextScreen extends java.awt.Canvas
{
    // Constants
    // TextScreen attributes
    public static final int KEYECHO = 0x01; // Echo keyboard input?
    public static final int CURSOR = 0x02; // Display cursor?
    public static final int AUTOWRAP = 0x04; // Autowrap lines?
    // Modes, used to prevent excessive redraws, etc.
    private static final int WRITE = 1; // Not reading
    private static final int READLINE = 2; // In readLine mode
    private static final int READCHAR = 3; // In readChar mode

    // Private variables
    private int rows, cols; // Current rows and columns
    private int attributes; // Current attributes
    private int readMode; // Current read mode
    private StringBuffer readString; // String being read
    private int readTerminator; // Key that terminated the last READLINE
    private int curReadChar; // Character read for readChar
    private Image offscreen; // Image used for offscreen drawing
    private int cursorX, cursorY; // Cursor position in peer units
    private int curRow, curCol; // Current row and column, in characters
    private int fontHeight, fontMaxWidth, fontAscent; // Height and max width of cur. font
    private FontMetrics curFontMetrics; // Current font metrics
    private Font curFont; // Current font
    private boolean rvsMode; // In reverse video mode?
    private Rectangle curRegion; // Current region
    private Vector terminators; // Vector of Event characters that can terminate a READ.
    
    // Methods
    // Constructor

    /** Creates a TextScreen with the specified colors, fonts,
        attributes, rows, columns.

        @param initialFont The initial font to be used.
        @param bgColor The background color.
        @param fgColor The foreground color.
        @param initialRows The number of rows of text.
        @param initialCols The number of columns of text.
        @param attrs The initial attributes.
    */
    public TextScreen(Font initialFont, Color bgColor, Color fgColor,
        int initialRows, int initialCols, int attrs)
    {
        super();

        rows = initialRows;
        cols = initialCols;
        attributes = attrs;
        setBackground(bgColor);
        setForeground(fgColor);
        setFont(initialFont);
        rvsMode = false;
        readMode = WRITE;

        // We can only set this once if we can't resize the window.
        curFontMetrics = getFontMetrics(initialFont);
        fontHeight = curFontMetrics.getHeight();
        fontAscent = curFontMetrics.getAscent();
        fontMaxWidth = 0;
        int widths[] = curFontMetrics.getWidths();
        for(int i=0;i<256;i++)
            if (widths[i] > fontMaxWidth)
                fontMaxWidth = widths[i];
        
        // Region starts off as the entire screen
        curRegion = new Rectangle(0,0,cols,rows);
        
        // CR is always a terminating character
        terminators = new Vector();
        terminators.addElement(new Integer(13));
    }


    // Public Instance Methods

    /** Sets the terminators (characters which terminate a readLine
        call) to the characters in the given vector.
    */
    public void setTerminators(Vector chars)
    {
        terminators = (Vector)chars.clone();
    }
    
    /** Resizes the TextScreen to fit the current font, rows and
        columns.  Also resets region to the whole screen.
    */
    public void resizeToFit()
    {
        int width, height;
        int fontWidths[];

        // Compute the size and resize the component
        height = fontHeight * rows;
        width = fontMaxWidth * cols;
        resize(width,height);
        curRegion = new Rectangle(0,0,cols,rows);
        
        // Get a new offscreen drawing area
        Dimension s = size();
        offscreen = createImage(s.width,s.height);
        clearScreen();
    }

    /** Clears the region and homes the cursor. */
    public void clearScreen()
    {
        Graphics g = offscreen.getGraphics();
        g.setColor(getBackground());
        g.fillRect(curRegion.x*fontMaxWidth,curRegion.y*fontHeight,
                   curRegion.width*fontMaxWidth,curRegion.height*fontHeight);
        curRow = curRegion.y;
        curCol = curRegion.x;
        cursorX = curCol * fontMaxWidth;
        cursorY = curRow * fontHeight;
        g = getGraphics();
        paint(g);
    }

    /** Set the region */
    public void setRegion(Rectangle region)
    {
        curRegion = region;
    }
    
    /** Print a string, wrapping as necessary.  Affected by region. */
    public void printString(String str)
    {
        Dimension s = size();
        Graphics g = offscreen.getGraphics();
        int maxRow, maxCol;
        
        maxCol = curRegion.x + curRegion.width;
        maxRow = curRegion.y + curRegion.height;
        
        g.setColor(getForeground());
        g.setFont(curFont);

        // First, tokenize this string into strings and newlines,
        // for maximum efficiency.
        StringTokenizer st = new StringTokenizer(str,"\b\n\r",true);

        // Now process a token at a time.
        while (st.hasMoreTokens()) {
            String token = st.nextToken();
            // See if this is a newline--if so, process it
            if (token.equals("\n")) {
                cursorX = curRegion.x * fontMaxWidth;
                curCol = curRegion.x;
                if ((curRow + 1) == maxRow)
					scrollUp(1);
                curRow++;
                cursorY = cursorY + fontHeight;
                continue;
            }
            // If this is a backspace, do a backspace
            if (token.equals("\b")) {
                if (curCol > curRegion.x)
                    gotoXY(curCol-1,curRow);
                else if (curRow > curRegion.y)
                    gotoXY(cols-1,curRow-1);
                g.clearRect(cursorX,cursorY,fontMaxWidth,fontHeight);
                continue;
            }
			// If this is a CR (not newline), go to the beginning of the line.
			if (token.equals("\r")) {
				gotoXY(curRegion.x,curRow);
				continue;
			}
            // Otherwise, print it.  If we are wrapping, do appropriate
            // checks.  This is still fairly crude and may look funny
            // in proportional fonts.
            if ((attributes & AUTOWRAP) == AUTOWRAP) {
                while ((curCol + token.length()) >= maxCol) {
                    // Draw what we can, do a newline, reduce
                    // the string.
                    String sub = token.substring(0,(maxCol - curCol));
                    if (rvsMode) {
                        g.fillRect(cursorX,cursorY,curFontMetrics.stringWidth(sub),fontHeight);
                        g.setColor(getBackground());
                    }
                    else
                        g.clearRect(cursorX,cursorY,curFontMetrics.stringWidth(sub),fontHeight);
                    g.drawString(sub,cursorX,cursorY+fontAscent);
                    if (rvsMode)
                        g.setColor(getForeground());
                    token = token.substring(maxCol - curCol);
                    cursorX = curRegion.x * fontMaxWidth;
                    curCol = curRegion.x;
                    if ((curRow + 1) == maxRow) // We need to scroll.
						scrollUp(1);
                    curRow++;
                    cursorY = cursorY + fontHeight;
                 }
            }
            // Draw the last bit of the string (or the
            // entire string, if we haven't worried about word-
            // wrapping.
            if (rvsMode) {
                g.fillRect(cursorX,cursorY,curFontMetrics.stringWidth(token),fontHeight);
                g.setColor(getBackground());
            }
            else
                g.clearRect(cursorX,cursorY,curFontMetrics.stringWidth(token),fontHeight);
            g.drawString(token,cursorX,cursorY+fontAscent);
            if (rvsMode)
                g.setColor(getForeground());
            curCol += token.length();
            cursorX += curFontMetrics.stringWidth(token);
        }
        g = getGraphics();
        paint(g);
    }

    /** Do a newline--equivalent to printString("\n") -- affected by region */
    public void newline()
    {
        cursorX = curRegion.x * fontMaxWidth;
        curCol = curRegion.x;
        cursorY = cursorY + fontHeight;
        curRow++;

        // Scroll if necessary (curRow et al are adjusted by scrollUp)
        if (curRow == (curRegion.x + curRegion.height))
            scrollUp(1);
        else {
            Graphics g = getGraphics();
            paint(g);
        }
    }

    /** Go to the specified X/Y coordinates (specified in characters).
        May look strange in proportional fonts.  NOT affected by region.

        @param x The x coordinate to go to.
        @param y The y coordinate to go to.
    */
    public void gotoXY(int x,int y)
    {
        if (x < cols)
            curCol = x;
        else
            curCol = cols - 1;
        if (y < rows)
            curRow = y;
        else
            curRow = rows - 1;

        cursorX = curCol * fontMaxWidth;
        cursorY = curRow * fontHeight;
    }

    /** Scroll the region up by the specified number of lines.
        If the number of lines is greater than the number of lines
        in the region, the screen will be cleared.

        @param numLines The number of lines to scroll
    */
    public void scrollUp(int numLines)
    {
        // First make sure this makes sense
        if (numLines < 1)
            return;
        if (numLines >= curRegion.height) {
            clearScreen();
            return;
        }

        // OK, do it by copying a region of the drawing area.
        Graphics g = offscreen.getGraphics();
        Dimension s = size();
        int scrollSize = numLines*fontHeight;
        g.copyArea(curRegion.x * fontMaxWidth,
                   (curRegion.y * fontHeight) + scrollSize,
                   curRegion.width * fontMaxWidth,
                   (curRegion.height * fontHeight) - scrollSize,
                   0,-(scrollSize));
        g.setColor(getBackground());
        g.fillRect(curRegion.x * fontMaxWidth,(((curRegion.y + curRegion.height) * fontHeight)-scrollSize),
                   curRegion.width * fontMaxWidth,scrollSize);
        g.setColor(getForeground());
        curRow = curRow - numLines;
        cursorY = curRow * fontHeight;
    }

    /** Put the TextScreen into or out of reverse video mode.

        @param mode true for reverse video on, false for reverse video off
    */
    public void reverseVideo(boolean mode)
    {
        rvsMode = mode;
    }

    // ADD the specified text style to the styles for this font,
    // unless it's Font.PLAIN, in which case everything goes back
    // to normal.
    public void setFontStyle(int style)
    {
        Font f = curFont;
        int newstyle;
        
        if (style == Font.PLAIN)
            newstyle = Font.PLAIN;
        else
            newstyle = (f.getStyle() | style);

        setFont(new Font(f.getName(),newstyle,f.getSize()));
    }
    
	/** Return the current position (in characters) of the cursor.
	*/
	public Point getCursorPosition()
	{
		Point d;

		d = new Point(curCol,curRow);
		return d;
	}

    /** Return the current size (in characters) of the screen.
    */
    public Dimension getSize()
    {
        Dimension d;

        d = new Dimension(cols,rows);
        return d;
    }

    /** Set the current size (in characters) of the screen.  Clear screen.
    */
    public synchronized void setSize(int width,int height)
    {
		rows = height;
		cols = width;
		resizeToFit();
    }

	/** Set the current size (in characters) of the screen.  Don't clear
		screen; if saveBottom is true, save the bottom portion of the screen
		if the screen is shrinking; otherwise, save the top portion.
	*/
	public synchronized void setSize(int width,int height,boolean saveBottom)
	{
		Image oldScreen;
		Rectangle r = new Rectangle();
		int oldRows, oldCols;

		// First, save the appropriate portion of the screen.
		if (height < rows) {
			if (saveBottom) {
				r.x = 0;
				r.y = (rows * fontHeight) - (height * fontHeight);
				r.width = (width * fontMaxWidth);
				r.height = (height * fontHeight);
			}
			else {
				r.x = 0;
				r.y = 0;
				r.width = (width * fontMaxWidth);
				r.height = (height * fontHeight);
			}
		}
		else { // Making screen bigger, so copy the whole thing
			r.x = 0;
			r.y = 0;
			r.width = size().width;
			r.height = size().height;
		}

		// Copy the screen portion
		CropImageFilter filter = new CropImageFilter(r.x,r.y,r.width,r.height);
		oldScreen = getToolkit().createImage(new FilteredImageSource(offscreen.getSource(),filter));

		// Now resize the component
		oldRows = rows;
		oldCols = cols;
		rows = height;
		cols = width;

        // Compute the size and resize the component
        height = fontHeight * rows;
        width = fontMaxWidth * cols;
        resize(width,height);

        // Get a new offscreen drawing area
        Dimension s = size();
        offscreen = createImage(s.width,s.height);

		// Clear it, then put the old image data on it
        Graphics g = offscreen.getGraphics();
        g.setColor(getBackground());
        g.fillRect(0,0,s.width,s.height);
		g.drawImage(oldScreen,0,0,this);

		// Place the cursor in an appropriate place--adjust it if necessary to
		// keep it in the same place in the text; if it is offscreen, home it.
		curRow -= (oldRows - rows);
		if ((curRow < 0) || (curRow >= rows)) {
		    curCol = 0;
		    curRow = 0;
		}
		cursorX = curCol * fontMaxWidth;
		cursorY = curRow * fontHeight;
		
		// Paint it.
		g = getGraphics();
		paint(g);
	}

	/** Get the size of the current font.
	*/
	public Dimension getFontSize()
	{
		Dimension d = new Dimension();

		d.width = curFontMetrics.getMaxAdvance();
		d.height = curFontMetrics.getHeight();
		return d;
	}

	/** Set the attributes after instantiation.
	*/
	public void setAttributes(int attrs)
	{
		attributes = attrs;
	}

	/** Get the current attributes
	*/
	public int getAttributes()
	{
		return attributes;
	}


    /** Read a line of text from the keyboard and append it to a
        supplied StringBuffer.  Return the character that terminated
        the operation.  If KEYECHO is on, echo it as it is input.
        A return value of -1 implies a timeout.  A return value of
        -2 implies an error.
    */
    public synchronized int readLine(StringBuffer sb)
    {
        Graphics g = getGraphics();

        if ((attributes & CURSOR) == CURSOR) {
            // Draw the cursor, in case it disappeared
            g.setColor(getForeground());
            g.fillRect(cursorX+1,cursorY+1,fontMaxWidth-2,fontHeight-2);        
        }
        readString = sb;
        readMode = READLINE;
		try {
	        wait();
		}
		catch (InterruptedException ex) {
			return -2;
		}

        // When we resume, we'll have the string!  Magic!
        if ((attributes & CURSOR) == CURSOR)
            g.clearRect(cursorX+1,cursorY+1,fontMaxWidth-2,fontHeight-2);        
        return readTerminator;
    }

    /** As readLine(StringBuffer), except that that the read will
        time out after time milliseconds.  The return value is
        -1 if a timeout occurred.
    */
    public synchronized int readLine(StringBuffer sb,long time)
    {
        Graphics g = getGraphics();
        boolean timedOut = false;
        
        if ((attributes & CURSOR) == CURSOR) {
            // Draw the cursor, in case it disappeared
            g.setColor(getForeground());
            g.fillRect(cursorX+1,cursorY+1,fontMaxWidth-2,fontHeight-2);        
        }
        readString = sb;
        readMode = READLINE;
		try {
	        wait(time);
		}
		catch (InterruptedException ex) {
			return -2;
		}

        // If readMode is still READLINE, that means we timed out.
        if (readMode == READLINE) {
            System.out.println("Timeout!");
            readMode = WRITE;
            timedOut = true;
        }
        
        // Nuke the cursor
        if ((attributes & CURSOR) == CURSOR)
            g.clearRect(cursorX+1,cursorY+1,fontMaxWidth-2,fontHeight-2);        
        
        // Return an appropriate value
        if (timedOut)
            return -1;
        else
            return readTerminator;
    }

    /** Read a character from the keyboard and return it.  If KEYECHO
        is on, echo it.
    */
    public synchronized int readChar()
    {
        Graphics g = getGraphics();
        
        if ((attributes & CURSOR) == CURSOR) {
            g.setColor(getForeground());
            g.fillRect(cursorX+1,cursorY+1,fontMaxWidth-2,fontHeight-2);        
        }
        readMode = READCHAR;
		try {
	        wait();
		}
		catch (InterruptedException ex) {
			return 0;
		}

        // When we resume, we'll have the character.
        if ((attributes & CURSOR) == CURSOR)
            g.clearRect(cursorX+1,cursorY+1,fontMaxWidth-2,fontHeight-2);        
        return curReadChar;
    }

    /** Read a character from the keyboard and return it, timing out after
        time milliseconds.  Return the character or -1 if a timeout occurs.
    */
    public synchronized int readChar(long time)
    {
        Graphics g = getGraphics();
        boolean timedOut = false;
        
        if ((attributes & CURSOR) == CURSOR) {
            g.setColor(getForeground());
            g.fillRect(cursorX+1,cursorY+1,fontMaxWidth-2,fontHeight-2);        
        }
        readMode = READCHAR;
		try {
	        wait(time);
		}
		catch (InterruptedException ex) {
			return -2;
		}

        // If readMode is still READCHAR, we timed out.
        if (readMode == READLINE) {
            readMode = WRITE;
            timedOut = true;
        }
        
        // Get rid of the cursor
        if ((attributes & CURSOR) == CURSOR)
            g.clearRect(cursorX+1,cursorY+1,fontMaxWidth-2,fontHeight-2);
        
        // Return the appropriate value
        if (timedOut)
            return -1;
        else
            return curReadChar;
    }

    /** Override of keyDown(). */
    public synchronized boolean keyDown(Event e,int key)
    {
        switch (readMode) {
        case READLINE : // We're reading a line
            if (terminators.contains(new Integer(key))) {
                readMode = WRITE;
                readTerminator = key;
                if (((char)key == '\n') && ((attributes & KEYECHO) == KEYECHO))
                    printString("\n");
                notify();
                return true;
            }
            if (((attributes & KEYECHO) == KEYECHO) && !((((char)key) == '\b') && (readString.length() == 0))) // Mmmm...hacky
                printString(String.valueOf((char)key));
            switch ((char)key) {
                case '\b' : if (readString.length() > 0)
                                readString.setLength(readString.length()-1);
                            break;
                default : readString.append(String.valueOf((char)key));
                          break;
            }
            return true;
        case READCHAR : curReadChar = key;
                        if ((attributes & KEYECHO) == KEYECHO)
                            printString(String.valueOf((char)curReadChar));
                        readMode = WRITE;
                        notify();
                        break;
        case WRITE : return false; // Weird.
        }

		return false;
    }

    /** Override of paint() method. */
    public void paint(Graphics g)
    {
        // Just copy the offscreen drawing area to the screen,
        // and optionally draw a cursor.
		if (offscreen != null)
			g.drawImage(offscreen,0,0,this);
        if (((attributes & CURSOR) == CURSOR) && (readMode != WRITE)) {
            g.setColor(getForeground());
            g.fillRect(cursorX+1,cursorY+1,fontMaxWidth-2,fontHeight-2);
         }
    }

    /** Override of setFont() - gathers some information about
        the font.
    */
    public synchronized void setFont(Font f)
    {
        curFont = f;
//        super.setFont(f);  // Not needed??

// We can't change font metrics if we can't resize the window
//        curFontMetrics = getFontMetrics(f);
//        fontHeight = curFontMetrics.getHeight();
//        fontAscent = curFontMetrics.getAscent();
//        fontMaxWidth = 0;
//        int widths[] = curFontMetrics.getWidths();
//        for(int i=0;i<256;i++)
//            if (widths[i] > fontMaxWidth)
//                fontMaxWidth = widths[i];
    }

    /** Override of minimumSize() */
    public Dimension minimumSize()
    {
        return(new Dimension((cols * fontMaxWidth),(rows * fontHeight)));
    }

    /** Overrise of preferredSize() */
    public Dimension preferredSize()
    {
        return(minimumSize());
    }

    /** We override addNotify() so we can do a few things as soon
        as our peer is created.
    */

    public void addNotify()
    {
        super.addNotify();

        resizeToFit(); // Also creates an offscreen drawing area,
                       // clears screen.
    }
}
