/* UnQuill: Disassemble games written with the "Quill" adventure game system
    Copyright (C) 1996-2000  John Elliott

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

                             - * -

    Written for gcc under Unix; compiles with Hi-Tech C under CP/M and 
    Pacific C under DOS.
 
    Under CP/M, compile with:  C -DTINY UNQUILL.C TABLES.C CONDACT.C INFORM.C

*/


/* Pull Quill data from a .SNA file.  *
 * Output to stdout.                  */

/* Format of the Quill database:

 Somewhere in memory (6D04h for vanilla Quill with no Illustrator/Patch/Press)
there is a colour definition table:

	DEFB	10h,ink,11h,paper,12h,flash,13h,bright,14h,inverse,15h,over
	DEFB	border

We use this as a magic number to find the Quill database in memory. After
the colour table is a table of values:

	DEFB	no. of objects player can carry at once
	DEFB	no. of objects
	DEFB	no. of locations
	DEFB	no. of messages

then a table of pointers:

	DEFW	rsptab	;Response table
	DEFW	protab	;Process table
	DEFW	objtop	;Top of object descriptions
	DEFW	loctop	;Top of locations.
	DEFW	msgtop	;Top of messages.
	**
	DEFW	conntab	;Table of connections.
	DEFW	voctab	;Table of words.
	DEFW	oloctab	;Table of object inital locations.
	DEFW	free	;[Early] Start of free memory
                    ;[Late]  Word/object mapping
	DEFW	freeend	;End of free memory
	DEFS	3	;?
	DEFB	80h,dict ;Expansion dictionary (for compressed games).
			         ;Dictionary is stored in ASCII, high bit set on last
			         ;letter of each word.
rsptab:	...		     ;response	- both stored as: DB verb, noun
protab: ...          ;process                     DW address-of-handler
                     ;                               terminated by verb 0.
                                         Handler format is:
                                          DB conditions...0FFh
                                          DB actions......0FFh
objtab:	...		;objects	- all are stored with all bytes
objtop:	DEFW	objtab        complemented to deter casual hackers.
        DEFW    object1
        DEFW    object2 etc.
loctab:	...		;locations       Texts are terminated with 0E0h
loctop:	DEFW    loctab           (ie 1Fh after decoding).
        DEFW    locno1
        DEFW    locno2 etc.
msgtab:	...		;messages
msgtop:	DEFW    msgtab
        DEFW    mess1
        DEFW    mess2 etc.
conntab: ...    ;connections    - stored as DB WORD,ROOM,WORD,ROOM...
                                  each entry is terminated with 0FFh.

voctab: ...		;vocabulary     - stored as DB 'word',number - the
                                  word is complemented to deter hackers.
                                  Table terminated with a word entry all
                                  five bytes of which are 0 before
                                  decoding.
oloctab: ...    ;initial locations of objects. 0FCh => not created
                                               0FDh => worn
                                               0FEh => carried
                                               0FFh => end of table

In later games (those made with Illustrator?), there is an extra byte 
(the number of system messages) after the number of messages, and at 
the label ** the following word is inserted:

        DEFW    systab  ;Points to word pointing to table of system
                    ;messages.
systab: DEFW    sysm0
        DEFW    sysm1 etc.

The address of the user-defined graphics is stored at 23675. In "Early" games,
the system messages are at UDGs+168.

CPC differences:

* I don't know where the UDGs are.
* Strings are terminated with 0xFF (0 after decoding) rather than 0xE0 
  (0x1F).
* No Spectrum colour codes in strings. Instead, the 
  code 0x14 means "newline" and 0x09 means "toggle reverse video".
  There are other CPC colour codes in the range 0x01-0x0A, but I don't
  know their meaning.
* I have assumed that the database always starts at 0x1BD1, which it does 
  in the snapshots I have examined.
  
Commodore 64 differences:

* I don't know where the UDGs are.
* Strings are terminated with 0xFF (0 after decoding) rather than 0xE0 
  (0x1F).
* No Spectrum colour codes in strings. 
* I have assumed that the database always starts at 0x0804, which it does 
  in the snapshots I have examined.

  
*/


#define MAINMODULE
#include "unquill.h"    /* Function prototypes */

#if DOS
/* TODO: Make this support the SWITCHAR in DRDOS and MSDOS 2-3 */
#define isoptch(c) ( (c) == '/' || (c) == '-')
#endif

#if CPM
#define isoptch(c) ( (c) == '/' || (c) == '-' || (c) == '[')
#endif

#ifndef isoptch
#define isoptch(c) ( (c) == '-' )
#endif


static ushrt ucptr;

int main(int argc, char *argv[])
{
	ushrt n, zxptr, seekpos;
	char *ofptr = "";
	
/* If command looks like a help command, print helpscreen */

	if ((argc == 1)   || (strchr("-/[", AV10) && strchr("hH/?", AV11)))
		usage(AV0);
	
/* Parse for options. */

	for (n = 2;n < argc; n++)
	{
		if (isoptch(argv[n][0]))	/* Start of an option */
		{
			switch (argv[n][1])
      			{
      				case 'T': 
      				case 't': switch (argv[n][2])
          			{
          			        case 'Z':
					case 'z': oopt = 'Z'; break;

					case '5': oopt = '5'; break;

#ifndef TINY
					case 'I':
					case 'i': oopt = 'I'; break;
#endif
					case 'r':
					case 'R': if (ofarg) fprintf(stderr, "%s: -O present. Ignoring -TR\n",AV0);
     						  else
     						  {
     						  	oopt    = 'R';
							running = 1;
              					  }
            					  break;
					default:  fprintf(stderr,"%s: Incorrect -T option. Type %s -H for help \n",AV0, AV0);
            					  exit(1);
           			}
           			break;
           			
			        case 'O':
			        case 'o': if (running)
				{
					fprintf(stderr, "%s: -O present. Ignoring -TR\n",AV0);
					running = 0;
					oopt    = 'T';
          			}
				ofarg = n;
				ofptr = (argv[n]+2);
				break;

			        case 'L':
			        case 'l': dbver = 10; break;

				case 'C': 
				case 'c': copt++; break;

				case 'g': 
				case 'G': gopt++; break;

				case 'm':
				case 'M': mopt++; break;

			        case 'S': 
			        case 's': switch(argv[n][2])
				{
					case 'C': case 'c': skipc = 1; break;
					case 'G': case 'g': skipg = 1; break;
					case 'O': case 'o': skipo = 1; break;
					case 'M': case 'm': skipm = 1; break;
					case 'N': case 'n': skipn = 1; break;
					case 'L': case 'l': skipl = 1; break;
					case 'V': case 'v': skipv = 1; break;
					case 'S': case 's': skips = 1; break;
					case 'U': case 'u': skipu = 1; break;
				        default:
            				fprintf(stderr,"%s: Syntax error. -S only takes C,O,M,N,L,V,S,U\n",AV0);
            				exit(1);
            			}
			        break;
 
 				case 'V':
 				case 'v': verbose = 1; break;

				case 'Q':
				case 'q': nobeep = 1; break;
			
				default:
        			fprintf(stderr,"%s: Invalid option: %s\n",AV0,argv[n]);
        			exit(1);
        		}	/* End switch */
		}	/* End For */
    	}	/* End If */

  /* If output file was specified, select it */

 	if (ofarg)
	{
		char opt[3];
		
		if (oopt == '5') strcpy(opt, "wb");
		else		   strcpy(opt, "w");
		if ((outfile = fopen(ofptr, opt)) == NULL)
		{
			perror(ofptr);
			exit(1);
		}
	}
	else
	{
		outfile=stdout;
		if (oopt == '5' && !copt)
		{
			fprintf(stderr,"%s: Z-Code files are not written to "
                                       "standard output. If this is really\n"
                                       "what you want to do, use the -C "
                                       "option.\n\n", AV0);
			exit(0);
		}
	}
 /* Load the snapshot. To save space, we ignore the printer
  buffer and the screen, which can contain nothing of value. */

	inname = argv[1];

	if((infile = fopen(inname,"rb")) == NULL)
	{

/* Since perror() in the CP/M version of Hi-Tech C isn't terribly good, I'll
  sometimes make special arrangements for CP/M error reporting. */

#ifdef CPM
		fprintf(stderr,"%s: Cannot open file.\n",inname);
#else
		perror(inname);
#endif
		exit(1);
	}

/* << v0.7: Check for CPC6128 format */
	if (fread(snapid, sizeof(snapid), 1, infile) != 1)
	{
#ifdef CPM
		fprintf(stderr,"%s: Not in Spectrum, CPC or C64 snapshot format.\n",inname);
#else
		perror(inname);
#endif

	}

	arch       = ARCH_SPECTRUM;
	seekpos    = 0x1C1B;	/* Number of bytes to skip in the file */
	mem_base   = 0x5C00;	/* First address loaded */
	mem_size   = 0xA400;	/* Number of bytes to load from it */
	mem_offset = 0x3FE5;	/* Load address of snapshot in host system memory */
	
	if (!memcmp(snapid, "MV - SNA", 9))	/* CPCEMU snapshot */
	{
		arch = ARCH_CPC;

		seekpos    = 0x1C00;
		mem_base   = 0x1B00;
		mem_size   = 0xA500;
		mem_offset = -0x100;
		dbver = 10;	/* CPC engine is equivalent to  
                                 * the "later" Spectrum one. */

		fprintf(stderr,"CPC snapshot signature found.\n");
	}
	if (!memcmp(snapid, "VICE Snapshot File\032", 19))	/* VICE snapshot */
	{
		arch = ARCH_C64;

		seekpos    =  0x874;
		mem_base   =  0x800;
		mem_size   = 0xA500;
		mem_offset =  -0x74;
		dbver = 5;	/* C64 engine is between the two Spectrum 
                     		 * ones. */

		fprintf(stderr,"C64 snapshot signature found.\n");
	}
/* >> v0.7 */

/* Skip screen/printer buffer/registers and load the rest */

#ifndef TINY

	if (fseek(infile,seekpos,SEEK_SET))
	{
		perror(inname);
		exit(1);
	}

/* Load file */

	if (fread(zxmemory, mem_size, 1, infile) != 1)
	{
		perror (inname);
		exit(1);
	}

#endif  /* ndef TINY */

/* .SNA read ok. Find a Quill signature */

	switch(arch)
	{
		case ARCH_SPECTRUM:

		/* I could _probably_ assume that the Spectrum database is 
		 * always at one location for "early" and another for "late"
		 * games. But this method is safer. */
		
		    	for (n = 0x5C00; n < 0xFFF5; n++)
				{
					if ((zmem(n  ) == 0x10) && (zmem(n+2) == 0x11) && 
						(zmem(n+4) == 0x12) && (zmem(n+6) == 0x13) && 
						(zmem(n+8) == 0x14) && (zmem(n+10) == 0x15))
					{
						fprintf(stderr,"Quill signature found.\n");
						found = 1;
						zxptr = n + 13;
						break;
					}
				}
				break;

		case ARCH_CPC: 
		        found = 1;
				zxptr = 0x1BD1;	/* From guesswork: CPC Quill files
				                 * always seem to start at 0x1BD1 */
				break;
		case ARCH_C64: 
		        found = 1;
				zxptr = 0x804;	/* From guesswork: C64 Quill files
				                 * always seem to start at 0x804 */
				break;
	}
	
	if (!found)
	{
		fprintf(stderr,"%s does not seem to be a valid Quill .SNA file. If you\n",
          		inname);
		fprintf(stderr,"know it is, you have found a bug in %s.\n",AV0);
  		exit(1);
  	}

#ifdef TINY      /* Fill the first buffer */

	if (fseek(infile, zxptr - mem_offset, SEEK_SET))
	{
#ifdef CPM
		fprintf(stderr, "Can't fseek to %x in %s\n", zxptr - mem_offset,
			inname);
#else
		perror(inname);
#endif
		exit(1);
	}

/* Load first buffer (pointers & dictionary) */

	b1base = zxptr;
	if (fread(buf1, sizeof(buf1), 1, infile) != 1)
	{
#ifdef CPM
		fprintf(stderr, "Can't load buffer at %x from %s\n", 
					b1base, inname);
#else
		perror (inname);
#endif
		exit(1);
	}
	b1full++;

#endif  /* def TINY */

	ucptr   = zxptr;
	maxcar1 = maxcar = zmem (zxptr);	/* Player's carrying capacity */
	nobj             = zmem (zxptr +  1);	/* No. of objects */
	nloc             = zmem (zxptr +  2);	/* No. of locations */
	nmsg             = zmem (zxptr +  3);	/* No. of messages */
	if (dbver)
 	{
  		++zxptr;
		nsys     = zmem (zxptr +  3);	/* No. of system messages */
		vocab    = zword(zxptr + 18);	/* Words list */
  		dict     = zxptr + 29;		/* Expansion dictionary */
	}
	else vocab = zword(zxptr + 16);

#ifdef TINY      /* Fill the second buffer */

	if (fseek(infile, vocab - mem_offset, SEEK_SET))
	{
#ifdef CPM
                fprintf(stderr, "Can't fseek to %x in %s\n", vocab - mem_offset,
                        inname);
#else
                perror(inname);
#endif
		exit(1);
	}

/* Load second buffer (vocabulary) */

	b2base = vocab;
	if (fread(buf2, sizeof(buf2), 1, infile) != 1)
	{
#ifdef CPM
                fprintf(stderr, "Can't load buffer at %x from %s\n", 
				b2base, inname);
#else
                perror (inname);
#endif
		exit(1);
	}
	b2full++;

#endif  /* def TINY */
	resptab            = zword(zxptr +  4);
        proctab            = zword(zxptr +  6);
	objtab             = zword(zxptr +  8);
	loctab             = zword(zxptr + 10);
	msgtab             = zword(zxptr + 12);
	if (dbver) sysbase = zword(zxptr + 14);
  	else       sysbase = zword(23675) + 168; /* Just after the UDGs */ 
	conntab            = zword(zxptr + 14 + ( dbver ? 2 : 0));
	if(dbver) objmap   = zword(zxptr + 22);
	postab             = zword(zxptr + 18 + ( dbver ? 2 : 0 ));

	switch(oopt)
	{
		case 'T':	/* Text */ 
		case 'Z':	/* ZXML */
	
		fprintf(outfile,"%4x: Player can carry %d objects.\n\n", ucptr, maxcar);

		if (!skipc)
		{
			fprintf(outfile, "%4x: Response [Event] table\n", zword(zxptr + 4));
			listcond(resptab);
			fprintf(outfile, "%4x: Process [Status] table\n", zword(zxptr + 6));
			listcond(proctab);
		}
		if (!skipo) listitems("Object",   (zword(objtab)), nobj);
		if (!skipl) listitems("Location", (zword(loctab)), nloc);
		if (!skipm) listitems("Message",  (zword(msgtab)), nmsg);
		if (!skipn) listconn (             zword(conntab), nloc);
		if (!skipv) listwords(vocab);
		if (!skipo) listpos(postab, nobj);
		if ((!skipo) && (dbver >= 10)) listmap(objmap, nobj);

		if (!skips)
		{
			if (dbver > 0) listitems("System message", (zword(sysbase)), nsys);
			 else       listitems("System message", sysbase, 32);
		}
		if (!skipu && arch == ARCH_SPECTRUM) listudgs(zword(23675));
		if (!skipg && arch == ARCH_SPECTRUM && dbver > 0) listgfx();

/** WARNING ** gotos here */

closeout:
		if(fclose(infile))
		{
#ifdef CPM
			fprintf (stderr,"%s: cannot close file.\n",inname);
#else
			perror (inname);
#endif
			exit(1);
		}

		if (ofarg && fclose(outfile))
		{
			perror(ofptr);
			exit(1);
		}
		break;

		case '5':
		zcode_binary();
		goto closeout;	/** WARNING ** Uses goto */
#ifndef TINY	
		case 'I':	/* Inform source */
		inform_src(zxptr);
		goto closeout;	/** WARNING ** Uses goto */	
#endif	
		case 'R':	/* Run the game */
		
		ramsave[0x100] = 0; /* No position RAMsaved */
		while(running)
		{
			estop = 0;   
			srand(1);
			oopt  = 'T';        /* Outputs in plain text */
			initgame(zxptr); /* Initialise the game */
			playgame(zxptr); /* Play it */ 
			if (estop)
			{
				estop=0;	/* Emergency stop operation, game restarts */
	    			continue;   /* automatically */
    			}
			sysmess(13);
			opch32('\n');
			if (yesno()=='N')
			{
				running=0;
				sysmess(14);
			}
		}
    		putchar('\n');
    		break;
	}	/* End switch */
	return 0;
}	/* End main() */




void usage(char *title)
{
  fprintf(stderr,"UnQuill v0.8 - John Elliott, 3 February 2001\n\n");
  fprintf(stderr,"Command format is %s input-file {-Ooutput-file} {-opt -opt ... }\n\n",title);
  fprintf(stderr,"  Decompiles .SNA snapshots of Quilled games to text.\n");
  fprintf(stderr,"Options are:\n");
  fprintf(stderr,"-TR: Run the game!\n");
#ifndef TINY
  fprintf(stderr,"-TI: Output a source file for Inform\n");
#endif
  fprintf(stderr,"-T5: Output a Z5 file for Z-code interpreters\n");
  fprintf(stderr,"-TZ: Outputs text in ZXML (converts Spectrum colour/\n");
  fprintf(stderr,"     flash controls to HTML-style <ATTR> commands).\n");

#if (DOS || CPM) 
  fprintf(stderr, "[more]"); fflush(stderr); getch(); fprintf(stderr, "\n");
#endif
  fprintf(stderr,"-C  : Force output of Z-code to standard output\n");
  fprintf(stderr,"-G  : Make the Z-code file use graphical drawing characters\n");
  fprintf(stderr,"-M  : Make the Z-code file not use colours\n");
  fprintf(stderr,"-L  : Attempt to interpret as a 'later' type Quill file.\n");
  fprintf(stderr,"-O  : Redirect output to a file. If this option is not\n");
  fprintf(stderr,"     present, stdout is used.\n");
  fprintf(stderr,"-Sx : Skip section x. x can be one of:\n");
  fprintf(stderr,"     C - condition tables\n");
  fprintf(stderr,"     O - object texts\n");
  fprintf(stderr,"     M - message texts\n");
  fprintf(stderr,"     N - coNNections\n");
  fprintf(stderr,"     L - location texts\n");
  fprintf(stderr,"     V - vocabulary tables\n");
  fprintf(stderr,"     S - system messages\n");
  fprintf(stderr,"     U - User-defined graphics\n");
  fprintf(stderr,"     G - location graphics\n");
  fprintf(stderr,"-V : Verbose. Annotate condition and connection tables\n");
  fprintf(stderr,"    with message, object and location texts.\n");
  fprintf(stderr,"-Q : Quiet (no beeping)\n");
  fprintf(stderr,"Mail bug reports to <jce@seasip.demon.co.uk>.\n");
  exit(1);
}



void initgame(ushrt zxptr)
{
	uchr  n; 
	ushrt obase;

	alsosee = 1;	/* Options possibly set by later games */
	fileid  = 255;
	maxcar1 = maxcar;

	obase = postab; /* Object initial positions table */
	for (n = 0; n < 36; n++) flags[n] = 0;
	
	NUMCAR = 0;
	for (n = 0; n < 255; n++)
	{
		objpos[n] = zmem(obase + n);
		if (objpos[n] == 254) NUMCAR++;
	}
}


void savegame(void)
{
	uchr xor = fileid, l, n;
	FILE *savefp;
	char filename[255];
	
	for (n = 0; n < 37;   n++) xor ^= flags[n];
	for (n = 0; n < 0xDB; n++) xor ^= objpos[n];
	opch32('\n');
	printf("Save game to file>");
	fflush(stdout);
	fgets(filename,255,stdin);
	l = strlen(filename)-1;
	if ((filename[l] == '\r') || (filename[l] == '\n')) filename[l]=0;

	savefp = fopen(filename,"wb");
	if (!savefp)
	{
		printf("Could not create %s\n",filename);
		return;
	}
	if ((fputc(2,      savefp) == EOF)
	||  (fputc(1,      savefp) == EOF)
	||  (fputc(fileid, savefp) == EOF)
	||  (fwrite(flags, 1,37,  savefp) < 37)
	||  (fwrite(objpos,1,0xDB,savefp) < 0xDB)
	||  (fputc(xor,savefp)     == EOF)
	||  (fclose(savefp)))
	{
		fclose (savefp);
		printf("Write error on %s\n",filename);
		return;
	}
}

/* Format of Quill save file (based on a Spectrum .TAP file):

DW 0102h	;Length / magic no.
DB 0FFh		;Block type ("fileid") - usually 0FFh, but can be changed in
		;later games by PAUSE, subfunction 22.
DS 31		;Flags 0-30
DW xx		;Flags 31-32 = no. of turns
DB xx		;Flag  33    = verb
DB xx		;Flag  34    = noun
DW xx		;Flags 35-36 = location (flag 36 is always 0)
DS 0DBh		;Object locations table, terminated with 0FFh.
DB xsum		;XOR of all bytes except the 0102h.
*/

void loadgame(void)
{
	uchr xor=0,l;
	ushrt n;
	FILE *loadfp;
	char filename[255];
	uchr savefile[0x104];
	
	opch32('\n');
	printf("Load game from file>");
	fflush(stdout);
	fgets(filename,255,stdin);
	l = strlen(filename)-1;
	if ((filename[l] == '\r') || (filename[l] == '\n'))
		filename[l] = 0;

	loadfp = fopen(filename,"rb");
	if (!loadfp)
	{
		printf("Could not open %s\n",filename);
		return;
	}
	if ((fread(savefile,1,0x104,loadfp) < 0x104)
	|| (fclose(loadfp)))
	{
		fclose(loadfp);
		printf("Read error on %s\n",filename);
		return;
	}
	for (n = 2; n < 0x103; n++) xor ^= savefile[n];
	if ((xor !=savefile[0x103])
	||  (savefile[0] != 2)
	||  (savefile[1] != 1)
	||  (savefile[2] != fileid))
	{
		printf("%s is not a suitable save file\nPress RETURN...",filename);
		getch();
		return;
	}	
	memcpy(flags,  savefile + 3,  37);
	memcpy(objpos, savefile + 40, 0xDB);
}


char present(uchr obj)
{
	if ((objpos[obj]==254) || (objpos[obj]==253) || (objpos[obj]==CURLOC))
    		return 1;
  	return 0;
}



void playgame(ushrt zxptr)
{
	uchr desc = 1, verb, noun, pn, r, n; 
	char *lbstart;
	ushrt connbase;
	char linebuf[255];

	lbstart = linebuf;
	while(1) /* Main loop */
	{
		if (desc) 
		{
			clrscr();
			/* 0.7.5: Darkness */
			if (flags[0] && (!present(0))) 
			{
				sysmess(0);
				opch32('\n');
			}
			else
			{
				oneitem(loctab,CURLOC); 
				opch32('\n');
				listat(CURLOC);  /* List objects present */
			}
			desc = 0;

		/* Decrement flags depending on location descriptions */

			if (flags[2]) flags[2]--;
			if (flags[0] && flags[3]) flags[3]--;
			if (flags[0] && (!present(0)) && flags[4]) flags[4]--; 
		}

		/* [new in 0.7.0: Flag decrements moved to *after* the
                 * process table; this makes Bugsy work properly */

		/* Process "process" conditions */

		verb = 0xFF; noun = 0xFF;
		
		r = doproc(zword(zxptr + 6), verb, noun);  /* The Process Table */
		if      (r == 2) desc = 1;      /* DESC */
		else if (r == 3) break;  	/* END  */
		else 
		{ 
                   /* Decrement flags not depending on location descriptions */

                	if (flags[5]) flags[5]--;
                	if (flags[6]) flags[6]--;
                	if (flags[7]) flags[7]--;
                	if (flags[8]) flags[8]--;
       	        	if (flags[0] && flags[9]) flags[9]--;
	                if (flags[0] && (!present(0)) && flags[10]) flags[10]--;
 
			/* Print the prompt */ 

			pn = (rand() & 3);
			sysmess(pn + 2);
			opch32('\n');
			if (dbver == 0) sysmess(28);
			fflush(stdin);
			fgets(linebuf, 254, stdin);
			
			while (linebuf[0] == '*')  /* Diagnostics: */
			{
				if (linebuf[1]=='F')  /* *F: dump flags */
					for (n=0; n<36; n++) printf(" F%3.3d:%3.3d ",n,flags[n]);
				if (linebuf[1]=='O')  /* *O: dump object locations */
					for (n=0; n<255; n++) printf(" O%3.3d:%3.3d ",n,objpos[n]);
				fgets(linebuf,254,stdin);
			}
    			TURNLO++;
			if (TURNLO == 0) TURNHI++;

			/* Parse the input */
	
			lbstart = linebuf;
			verb    = matchword(&lbstart);
			if (verb != 0xFF)
 			{
 				VERB = verb;
				noun = matchword(&lbstart);
				NOUN = noun;
				if (noun == 0xFF) noun = 0xFE;

	/* v0.7.5: Moved "response" conditions to after the attempt to 
	 *         move the player
	 */

    	/* Attempt to move player */
				r = 0;
/*				if (verb < 20)*/   /* A movement word       */
/*				{  * This test is incorrect; Quill does not *
 *				   * insist on the word number being < 20   */
					connbase = zword(2*CURLOC+conntab);
					while ((!r) && (zmem(connbase) != 0xFF)) if (verb == zmem(connbase))
					{
						CURLOC=zmem(++connbase);
						desc = 1; r = 1;
					}
				   	else connbase += 2; 
	    			/* } */
    	 			if (r == 0)
				{
			        /* Process "response" conditions */
                                	r = doproc(zword(zxptr+4),verb,noun); 
                                	if      (r == 2) desc=1;        /* DESC */
                                	else if (r == 3) break;         /* END  */
				}
				/* Print "I can't do that/go that way" */
				if (r == 0)
				{
					if (verb < 20) sysmess(7); 
					else           sysmess(8);
					opch32('\n');
				}	
        		}
        		else sysmess(6); /* Unknown verb */
      		}
	}
}




uchr cplscmp(ushrt first, char *snd)
{
	if (((255-(zmem(first++))) & 0x7F) != (snd[0])) return 0;
	if (((255-(zmem(first++))) & 0x7F) != (snd[1])) return 0;
	if (((255-(zmem(first++))) & 0x7F) != (snd[2])) return 0;
	if (((255-(zmem(first++))) & 0x7F) != (snd[3])) return 0;
	return 1;
}




uchr matchword(char **wordbeg)
{
/* Match a word of player input with the vocabulary table */

	ushrt matchp=vocab;
	char wordbuf[5];
	int i;

	wordbuf[4]=0;
	while(1)
	{
		for (i=0; i<4; i++)
		{
			wordbuf[i]=(**wordbeg);
			if (islower(wordbuf[i])) wordbuf[i] = toupper(wordbuf[i]);	/* (v0.4.1), was munging numbers */
			if (wordbuf[i]==0)    wordbuf[i]=' ';
			if (wordbuf[i]=='\n') wordbuf[i]=' ';
			if (wordbuf[i]=='\r') wordbuf[i]=' ';
			if (wordbuf[i]==' ')  (*wordbeg)--;
			(*wordbeg)++;
		}
		while ((**wordbeg) 
		&&     (**wordbeg!='\n')
		&&     (**wordbeg!='\r')
		&&     (**wordbeg!=' '))
			(*wordbeg)++; 
		while (zmem(matchp))
		{
			if (cplscmp(matchp,wordbuf))
			{
				return zmem(matchp+4);
			}
			matchp+=5;
		}
		matchp=vocab;
		
		if (((**wordbeg)==0)
		||  ((**wordbeg)=='\r')
		||  ((**wordbeg)=='\n'))
		return 0xFF;

		while ((**wordbeg)==0x20)
		{
			if ((**wordbeg)==0) return 0xFF;
			(*wordbeg)++;
		}
	}
	return 0xFF;
		
}


void listat(uchr locno)
{
/* List items at location n */
	
	uchr any = 0, n;

	for (n=0; n<nobj; n++) if (objpos[n] == locno)
        {
		if (any==0)
		{
			any = 1;
			if (locno < 253) 
                  	{
				sysmess(alsosee);
				opch32('\n');
			} 
		}
		oneitem(objtab,n);
		opch32('\n');
	}
}



uchr ffeq(uchr x,uchr y)  /* Match x with y, 0FFh matches any */
{
	return (uchr)((x == 0xFF) || (y == 0xFF) || (x == y));
}




uchr doproc (ushrt table, uchr verb, uchr noun)
{
	ushrt ccond;
	uchr done = 0; /* Done returns: 0 for fell off end of table
                                        1 for DONE
                                        2 for DESC
                                        3 for END (end game) */
	uchr t, tverb, tnoun, td1 = 0;

	while ((zmem(table)) && !done)
    	{
		tverb = zmem(table++);
		tnoun = zmem(table++);
		ccond = zword(table++);
		table++;

		if (ffeq(verb,tverb) && ffeq(noun,tnoun))
		{
			t = condtrue(ccond);
      			/* Skip over condition clauses */
      			while (zmem(ccond++) != 0xFF);	
			if (t) 
			{
        			done = runact(ccond,noun);    
        			/* Returned nonzero if should not scan */
				td1 = 1;  /* Something was run */
			}
		} 
	}
	if ((done==0) && (td1)) done=1;
	return(done);
} 



uchr autobj(uchr noun)  /* Find object number for AUTOx actions */
{
	uchr n;

	if (dbver == 0) return 0xFF;
	if (noun > 253) return 0xFF;
	for (n = 0; n < nobj; n++) if (noun == zmem(objmap + n)) return n;
	return 0xFF;
}


char yesno(void)
{
	char n;
   
	fflush(stdin);
	fflush(stdout);
	
	while (1)
	{
		n = getchar();
		if ((n == 'Y') || (n == 'y')) return ('Y');
		if ((n == 'N') || (n == 'n')) return ('N');
	}
	return('N');
}



void sysmess(uchr sysno)
{
	uchr  cch = 0;
	ushrt msgadd;

	if (dbver > 0) 
	{	
		oneitem(sysbase,sysno);
		return;
	}
	msgadd = sysbase;
	while (sysno)	/* Skip (sysno) messages */
	{
		while (cch != 0x1F) cch = 0xFF - (zmem(msgadd++));
		sysno--;
		cch = 0xFF - (zmem(msgadd++));
      	}
	msgadd--;
	while (cch != 0x1F)
	{
		cch = 0xFF - (zmem(msgadd++));
		expch (cch,&msgadd);
	}
}




uchr zmem(ushrt addr)
{
/* All Spectrum memory accesses are routed through this procedure.
 * If TINY is defined, this accesses the .sna file directly.
 */

	if (addr < mem_base || (arch != ARCH_SPECTRUM && 
	                       (addr >= (mem_base + mem_size))))
	{
		fprintf (stderr,"\nInvalid address %4.4x requested. ", addr);
		if (arch != ARCH_SPECTRUM)
		     fprintf(stderr,"Probably not a Quilled game.\n");
		else fprintf(stderr, "Try %s the -L option.\n",
		                ((dbver > 0) ? "omitting":"including"));
		exit(1);
	}
#ifdef TINY

	if     ((b1full) && (addr >= b1base)  && (addr < (b1base + 0x400)))
    		return (buf1[addr-b1base]);
	else if ((b2full) && (addr >= b2base) && (addr < (b2base + 0x400)))
		return (buf2[addr-b2base]);
	else if (fseek(infile,addr - mem_offset,SEEK_SET))
	{
		perror(inname);
		exit(1);	
	}
  return (fgetc(infile));

#else

  return zxmemory[addr - mem_base];

#endif
}



ushrt zword(ushrt addr)
{
	return (ushrt)(zmem(addr) + (256 * zmem(addr + 1)));
}



void dec32(ushrt v)
{
	char decbuf[6];
#ifdef TINY
	char i;   /* 8-bit register on 8-bit computer */
#else
	int i;
#endif
	
	sprintf(decbuf,"%d",v);
	i=0;
	while (decbuf[i]) opch32(decbuf[i++]);
}



void opch32(char ch)	/* Output a character, assuming 32-column screen */
	{
	char x;

	fputc(ch, outfile);
	if   (ch == '\n')
	{
		xpos = 0;
		if (oopt == 'I' && comment_out) fputs("! ", outfile);
		if (indent)
		{
			for (x = 0; x < indent; x++) fputc(' ',outfile);
        		if (oopt != 'I') fputc(';',outfile);
        	}
      		else if (!running) fprintf(outfile,"      ");
                if (oopt == 'I') fputc('^', outfile);
    	}
    	else if (ch == 8) xpos--;
	else if (arch == ARCH_SPECTRUM && xpos == 31) opch32('\n');
	else if (arch != ARCH_SPECTRUM && xpos == 39) opch32('\n');
	else xpos++;
  }

#ifndef YES_GETCH


char getch(void)
{
	struct termios ts, ots;
        char c;
	int i;

	int filen = fileno(stdin);
        tcgetattr(filen, &ts);		/* Switch the terminal to raw mode */
        tcgetattr(filen, &ots);
        cfmakeraw(&ts);
        tcsetattr(filen, TCSANOW, &ts);
        
        fflush(stdout);
        fflush(stdin);
	do
	{
		i = read(filen, &c, 1);
	}
	while (i < 0);
        
        tcsetattr(filen, TCSANOW, &ots);

	fflush(stdin);
	return c;
}
#endif



