program parseRouteListings;
{ given a download from the CIM of routes, figure out the implied edges }

const
  maxSector = 1000;
  maxWarps  = 6;
type
  sectorIndex = 0..maxSector;
  warpIndex   = 0..maxWarps;
  sector = record
             warps : warpIndex;
             data  : array [1..maxWarps] of sectorIndex;
           end;
  space  = array [1..maxSector] of sector;

procedure OpenFile( var f : text; prompt, default : string; readOnly : boolean );
{ open text file up, typically with default value, for read or write }
var
  fname : string;
begin
  write( prompt, '[', default, ']  ');
  readln( fname );
  if fname = '' then
    fname := default;
  assign( f, fname );
  if readonly then
    reset( f )
  else
    rewrite( f );
end;

procedure InitSpace( var s : space );
{ assign all sectors with no outgoing warps }
var
  t : warpIndex;
  i : sectorIndex;
begin
  for i := 1 to maxSector do
    begin
      s[i].warps := 0;
      for t := 1 to maxWarps do
        s[i].data[t] := 0;
    end; {for i}
end;

procedure AddEdge( fromEdge, ToEdge : sectorIndex; var s : space );
{ add "toEdge" to list of warps at fromEdge, if new }
var
  t : warpIndex;
  found : boolean;
begin
  with s[ fromEdge ] do
    begin
      found := false;
      for t := 1 to warps do
        if data[t] = ToEdge then
          found := true;
      if not found then
        begin
          inc( warps );
          data[ warps ] := ToEdge;
        end; {if}
    end; {with}
end;

function stripLine( var s : string ) : integer;
{ peel number from front of s, then remove until '>' (if it exists).
  Return number, and updated s. }
var
  err : integer;
  n : integer;
  p : byte;
  l : string;
begin
  p := pos( '>', s );
  if p = 0 then
    begin
      l := s;
      s := '';
    end
  else
    begin
      l := copy( s, 1, p-2);
      s := copy( s, p+1, 255);
    end;
  val( l, n, err );
  stripLine := n;
end;

procedure ClearParens( var s : string );
{ remove any parentheses from the line }
var
  i : 0..255;
begin
  i := 1;
  while i <= length( s ) do
    if s[i] in ['(',')'] then
      s := copy( s, 1, i-1 ) + copy( s, i+1, 255)
    else
      inc( i );
end;

procedure parseLine( var f : text; var s : space );
{ the next line of f is a path.  Parse it, and load new data into s }
var
  current, previous : sectorIndex;
  line : string;
begin
  current := 0;
  readln( f, line );
  ClearParens( line );    { new for beta version 18 }
  while line <> '' do     { will terminate when a blank line is read }
    begin
      previous := current;
      current := stripLine( line );   {also updates line}
      if (previous <> 0) and (current <> 0) then
        AddEdge( previous, current, s );
      if line = '' then
        readln( f, line );
    end; {while}
end;

procedure ParseFile( var f : text; var s : space );
{ read file, and identify paths to be parsed }
var
  line : string;
begin
  while not eof( f ) do
    begin
      readln( f, line );
      if pos( 'TO', line ) = 3 then
        parseline( f, s );
    end;
end;

procedure StoreSpace( var f : text; var s : space );
{ given the data developed in s, write a pseudo .sct report to disk }
var
  i : sectorIndex;
  t : warpIndex;
begin
  for i := 1 to maxSector do
    begin
      write( f, i : 4 );
      if s[i].warps = 0 then
        writeln( f, '  0')
      else
        begin
          for t := 1 to s[i].warps do
            write( f, ' ', s[i].data[t]);
          writeln( f );
        end; {for else}
    end;
end;

var
  f : text;
  s : space;

begin {main}
  OpenFile( f, 'Logged file of warps? ', 'smc.rte', true );
  InitSpace( s );
  ParseFile( f, s );
  Close( f );
  OpenFile( f, 'File to store pseudo sector report? ', 'smc.sct', false );
  StoreSpace( f, s );
  close( f );
end.